<?php

namespace Noptin\Addons_Pack\Tasks;

/**
 * Contains the main task runner class.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class Task_Runner
 */
class Task_Runner {

	/**
	 * The cron hook.
	 */
	public $cron_hook = 'noptin_run_tasks';

	/**
	 * The start time.
	 *
	 * Represents when the queue runner was started.
	 *
	 * @var int
	 */
	protected $start_time;

	/**
	 * Class constructor.
	 */
	public function __construct() {

		add_filter( 'cron_schedules', array( $this, 'add_wp_cron_schedule' ), 100 );
		add_action( 'admin_init', array( $this, 'add_wp_cron_event' ) );
		add_action( $this->cron_hook, array( $this, 'run' ) );
	}

	public function add_wp_cron_schedule( $schedules ) {
		$schedules['every_minute'] = array(
			'interval' => 60, // in seconds
			'display'  => __( 'Every minute', 'noptin-addons-pack' ),
		);

		return $schedules;
	}

	/**
	 * Add cron workers
	 */
	public function add_wp_cron_event() {
		if ( ! wp_next_scheduled( $this->cron_hook ) ) {
			wp_schedule_event( time(), 'every_minute', $this->cron_hook );
		}
	}

	/**
	 * Runs the queue.
	 *
	 * Pass each queue item to the task handler, while remaining
	 * within server memory and time limit constraints.
	 */
	public function run() {

		// If already running, bail.
		if ( $this->is_process_running() ) {
			return 0;
		}

		// Raise the memory limit.
		wp_raise_memory_limit();

		// Raise the time limit.
		$this->raise_time_limit( $this->get_time_limit() + 10 );

		// Cleanup long running actions.
		Main::clean( 10 * $this->get_time_limit() );

		$processed_actions = 0;

		// Run the queue.
		do {
			$task = $this->get_next_task();

			if ( empty( $task ) ) {
				break;
			}

			// Lock process.
			if ( empty( $this->start_time ) ) {
				$this->lock_process();
			}

			if ( $task->has_expired() ) {
				$this->process_task( $task );
			}

			$processed_actions++;
		} while ( ! $this->batch_limits_exceeded( $processed_actions ) );

		// Unlock process.
		$this->unlock_process();

		// Clear the caches.
		$this->clear_caches();

		return $processed_actions;
	}

	/**
	 * Running large batches can eat up memory, as WP adds data to its object cache.
	 *
	 * If using a persistent object store, this has the side effect of flushing that
	 * as well, so this is disabled by default. To enable:
	 *
	 * add_filter( 'noptin_tasks_runner_flush_cache', '__return_true' );
	 */
	protected function clear_caches() {
		if ( ! wp_using_ext_object_cache() || apply_filters( 'noptin_tasks_runner_flush_cache', false ) ) {
			wp_cache_flush();
		}
	}

	/**
	 * Is process running
	 *
	 * Check whether the current process is already running
	 * in a background process.
	 */
	protected function is_process_running() {
		if ( get_transient( 'noptin_tasks_process_lock' ) ) {
			// Process already running.
			return true;
		}

		return false;
	}

	/**
	 * Lock process
	 *
	 * Lock the process so that multiple instances can't run simultaneously.
	 * Override if applicable, but the duration should be greater than that
	 * defined in the time_exceeded() method.
	 */
	protected function lock_process() {
		$this->start_time = time(); // Set start time of current process.

		$lock_duration = apply_filters( 'noptin_tasks_queue_lock_time', 60 ); // 1 minute.

		set_transient( 'noptin_tasks_process_lock', microtime(), $lock_duration );
	}

	/**
	 * Unlock process
	 *
	 * Unlock the process so that other instances can spawn.
	 *
	 * @return $this
	 */
	protected function unlock_process() {
		delete_transient( 'noptin_tasks_process_lock' );

		return $this;
	}

	/**
	 * Attempts to raise the PHP timeout for time intensive processes.
	 *
	 * Only allows raising the existing limit and prevents lowering it.
	 *
	 * @param int $limit The time limit in seconds.
	 */
	public static function raise_time_limit( $limit = 0 ) {
		$limit              = (int) $limit;
		$max_execution_time = (int) ini_get( 'max_execution_time' );

		/*
		 * If the max execution time is already unlimited (zero), or if it exceeds or is equal to the proposed
		 * limit, there is no reason for us to make further changes (we never want to lower it).
		 */
		if ( 0 === $max_execution_time || ( $max_execution_time >= $limit && 0 !== $limit ) ) {
			return;
		}

		if ( function_exists( 'set_time_limit' ) && false === strpos( ini_get( 'disable_functions' ), 'set_time_limit' ) && ! ini_get( 'safe_mode' ) ) { // phpcs:ignore PHPCompatibility.IniDirectives.RemovedIniDirectives.safe_modeDeprecatedRemoved
			@set_time_limit( $limit ); // @codingStandardsIgnoreLine
		}
	}

	/**
	 * Get the maximum number of seconds a batch can run for.
	 *
	 * @return int The number of seconds.
	 */
	protected function get_time_limit() {
		return absint( apply_filters( 'noptin_tasks_queue_runner_time_limit', 20 ) );
	}

	/**
	 * Check if the host's max execution time is (likely) to be exceeded if processing more actions.
	 *
	 * @param int $processed_actions The number of actions processed so far - used to determine the likelihood of exceeding the time limit if processing another action
	 * @return bool
	 */
	protected function time_likely_to_be_exceeded( $processed_actions ) {

		$execution_time        = time() - $this->start_time;
		$max_execution_time    = $this->get_time_limit();
		$time_per_action       = $execution_time / $processed_actions;
		$estimated_time        = $execution_time + ( $time_per_action * 3 );
		$likely_to_be_exceeded = $estimated_time > $max_execution_time;

		return apply_filters( 'noptin_tasks_maximum_execution_time_likely_to_be_exceeded', $likely_to_be_exceeded, $this, $processed_actions, $execution_time, $max_execution_time );
	}

	/**
	 * Get memory limit
	 *
	 * @return int
	 */
	protected function get_memory_limit() {
		if ( function_exists( 'ini_get' ) ) {
			$memory_limit = ini_get( 'memory_limit' );
		} else {
			// Sensible default.
			$memory_limit = '128M';
		}

		if ( ! $memory_limit || -1 === intval( $memory_limit ) ) {
			// Unlimited, set to 32GB.
			$memory_limit = '32G';
		}

		return wp_convert_hr_to_bytes( $memory_limit );
	}

	/**
	 * Memory exceeded
	 *
	 * Ensures the batch process never exceeds 90% of the maximum WordPress memory.
	 *
	 * Based on WP_Background_Process::memory_exceeded()
	 *
	 * @return bool
	 */
	protected function memory_exceeded() {

		$memory_limit    = $this->get_memory_limit() * 0.90;
		$current_memory  = memory_get_usage( true );
		$memory_exceeded = $current_memory >= $memory_limit;

		return apply_filters( 'noptin_tasks_memory_exceeded', $memory_exceeded, $this );
	}

	/**
	 * See if the batch limits have been exceeded, which is when memory usage is almost at
	 * the maximum limit, or the time to process more actions will exceed the max time limit.
	 *
	 * Based on WP_Background_Process::batch_limits_exceeded()
	 *
	 * @param int $processed_actions The number of actions processed so far - used to determine the likelihood of exceeding the time limit if processing another action
	 * @return bool
	 */
	protected function batch_limits_exceeded( $processed_actions ) {
		return $this->memory_exceeded() || $this->time_likely_to_be_exceeded( $processed_actions );
	}

	/**
	 * Retrieves the next task.
	 *
	 * @return false|Task Return the first batch from the queue
	 */
	protected function get_next_task() {

		$tasks = Main::query(
			array(
				'status'                => 'pending',
				'number'                => 1,
				'date_scheduled_before' => gmdate( 'Y-m-d H:i:s', time() + 30 ),
			)
		);

		return empty( $tasks ) ? false : $tasks[0];

	}

	/**
	 * Process an individual task.
	 *
	 * @param Task $task The task to process.
	 */
	public function process_task( $task ) {
		try {

			// Abort if the task is not pending.
			if ( 'pending' !== $task->get_status() ) {
				return;
			}

			do_action( 'noptin_tasks_before_execute', $task );

			// Mark the task as running.
			$task->task_started( $task );

			// Execute the task.
			$task->run();

			// Mark the task as complete.
			$task->task_complete( $task );

			do_action( 'noptin_tasks_after_execute', $task );
		} catch ( \Exception $e ) {
			$task->task_failed( $task, $e );
			do_action( 'noptin_tasks_failed_execution', $task, $e );
		}

	}
}
