<?php

namespace Noptin\Addons_Pack\Emails;

/**
 * Adds support for email attachments.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Adds support for email attachments.
 *
 */
class Attachments {

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		add_action( 'noptin_before_email_attachments_metabox', array( $this, 'display_attachment_metabox' ) );
		add_filter( 'noptin_send_email_args', array( $this, 'add_attachments_to_email' ) );
	}

	/**
	 * Registers the periodic email type.
	 *
	 * @param \Noptin_Automated_Email|\Noptin_Newsletter_Email $campaign
	 */
	public function display_attachment_metabox( $campaign ) {

		// Prepare attachments.
		$attachments = $campaign->get( 'attachments' );
		$attachments = ! is_array( $attachments ) ? array() : array_filter( $attachments );

		wp_nonce_field( 'noptin_email_attachments', 'noptin_email_attachments_nonce' );
		?>
			<ul id="noptin-addons-pack__attachments-container">
				<?php foreach ( $attachments as $attachment ) : ?>
					<li>
						<input type="text" class="widefat" name="noptin_email[attachments][]" placeholder="<?php esc_attr_e( 'Enter file path or URL', 'noptin-addons-pack' ); ?>" value="<?php echo esc_attr( $attachment ); ?>">
						<button class="button button-link noptin-addons-pack__upload-attachment">
							<span class="dashicons dashicons-upload"></span>
						</button>
						<button class="button button-link noptin-addons-pack__remove-attachment">
							<span class="dashicons dashicons-trash"></span>
						</button>
					</li>
				<?php endforeach; ?>
			</ul>

			<p class="description">
				<?php esc_html_e( 'You can only attach files that are hosted on your server.', 'noptin-addons-pack' ); ?>
			</p>

			<button class="button noptin-addons-pack__add-attachment">
				<?php esc_html_e( 'Add Attachment', 'noptin-addons-pack' ); ?>
			</button>

			<script>
				jQuery(document).ready(function($){
					var file_frame;

					// Delete attachment.
					$('#noptin-addons-pack__attachments-container').on('click', '.noptin-addons-pack__remove-attachment', function(e){
						e.preventDefault();
						$(this).parent().remove();
					});

					// Add attachment.
					$('.noptin-addons-pack__add-attachment').on('click', function(e){
						e.preventDefault();

						$( '#noptin-addons-pack__attachments-container' )
							.append( '<li><input type="text" class="widefat" placeholder="<?php echo esc_js( esc_attr__( 'Enter file path or URL', 'noptin-addons-pack' ) ); ?>" name="noptin_email[attachments][]"><button class="button button-link noptin-addons-pack__upload-attachment"><span class="dashicons dashicons-upload"></span></button><button class="button button-link noptin-addons-pack__remove-attachment"><span class="dashicons dashicons-trash"></span></button></li>' );
					});

					// Upload attachment.
					$('#noptin-addons-pack__attachments-container').on('click', '.noptin-addons-pack__upload-attachment', function(e){
						e.preventDefault();

						var input = $(this).parent().find('input');

						if (file_frame) {
							file_frame.open();
							return;
						}
						file_frame = wp.media.frames.file_frame = wp.media({
							title: 'Select or upload file',
							button: {
								text: 'Attach file'
							},
							multiple: false
						});
						file_frame.on('select', function() {
							var attachment = file_frame.state().get('selection').first().toJSON();
							input.val(attachment.url);
						});
						file_frame.open();
					});
				});
			</script>

			<style>
				#noptin-addons-pack__attachments-container li {
					margin-bottom: 10px;
					display: flex;
					flex-wrap: wrap;
				}

				#noptin-addons-pack__attachments-container li input {
					flex: 1;
				}

				/** Convert button to input group */
				#noptin-addons-pack__attachments-container li button {
					margin-left: 10px;
					text-decoration: none;
					color: currentColor;
				}
			</style>
		<?php
	}

	/**
	 * Adds attachments to the email.
	 *
	 * @param array $args
	 * @return array
	 */
	public function add_attachments_to_email( $args ) {

		// Abort if not sending an automated email.
		if ( empty( $args['campaign_id'] ) ) {
			return $args;
		}

		if ( 'automation' === get_post_meta( $args['campaign_id'], 'campaign_type', true ) ) {
			$campaign = new \Noptin_Automated_Email( (int) $args['campaign_id'] );
		} else {
			$campaign = new \Noptin_Newsletter_Email( (int) $args['campaign_id'] );
		}

		$attachments = $campaign->get( 'attachments' );

		if ( defined( 'NOPTIN_SENDING_TEST_EMAIL' ) && NOPTIN_SENDING_TEST_EMAIL && isset( $_POST['noptin_email_attachments_nonce'] ) && wp_verify_nonce( $_POST['noptin_email_attachments_nonce'], 'noptin_email_attachments' ) ) {
			$attachments = isset( $_POST['noptin_email']['attachments'] ) ? wp_parse_list( $_POST['noptin_email']['attachments'] ) : array();
		}

		$attachments = ! is_array( $attachments ) ? array() : array_filter( $attachments );

		// Abort if no attachments.
		if ( empty( $attachments ) ) {
			return $args;
		}

		// Prepare attachments.
		$args['attachments'] = isset( $args['attachments'] ) ? $args['attachments'] : array();

		foreach ( $attachments as $attachment ) {

			$attachment = $this->parse_attachment_file_path( trim( $attachment ) );

			// Add if its not a remote file.
			if ( ! $attachment['remote_file'] ) {
				$args['attachments'][] = $attachment['file_path'];
			}
		}

		return $args;
	}

	/**
	 * Parse file path/url and see if its remote or local.
	 *
	 * @param string $file_url
	 * @return array
	 */
	private function parse_attachment_file_path( $file_url ) {
		$wp_uploads     = wp_upload_dir();
		$wp_uploads_dir = $wp_uploads['basedir'];
		$wp_uploads_url = $wp_uploads['baseurl'];

		// Prevent path traversal.
		$file_url = str_replace( '../', '', $file_url );

		/**
		 * Replace uploads dir, site url etc with absolute counterparts if we can.
		 * Note the str_replace on site_url is on purpose, so if https is forced
		 * via filters we can still do the string replacement on a HTTP file.
		 */
		$replacements = array(
			$wp_uploads_url                  => $wp_uploads_dir,
			network_site_url( '/', 'https' ) => ABSPATH,
			str_replace( 'https:', 'http:', network_site_url( '/', 'http' ) ) => ABSPATH,
			site_url( '/', 'https' )         => ABSPATH,
			str_replace( 'https:', 'http:', site_url( '/', 'http' ) ) => ABSPATH,
		);

		$count            = 0;
		$file_path        = str_replace( array_keys( $replacements ), array_values( $replacements ), $file_url, $count );
		$parsed_file_path = wp_parse_url( $file_path );
		$remote_file      = null === $count || 0 === $count; // Remote file only if there were no replacements.

		// Paths that begin with '//' are always remote URLs.
		if ( '//' === substr( $file_path, 0, 2 ) ) {
			$file_path = ( is_ssl() ? 'https:' : 'http:' ) . $file_path;

			/**
			 * Filter the remote filepath for download.
			 *
			 * @since 1.0.0
			 * @param string $file_path File path.
			 */
			return array(
				'remote_file' => true,
				'file_path'   => apply_filters( 'hizzle_download_parse_remote_file_path', $file_path, true ),
			);
		}

		// See if path needs an abspath prepended to work.
		if ( file_exists( ABSPATH . $file_path ) ) {
			$remote_file = false;
			$file_path   = ABSPATH . $file_path;

		} elseif ( '/wp-content' === substr( $file_path, 0, 11 ) ) {
			$remote_file = false;
			$file_path   = realpath( WP_CONTENT_DIR . substr( $file_path, 11 ) );

			// Check if we have an absolute path.
		} elseif ( ( ! isset( $parsed_file_path['scheme'] ) || ! in_array( $parsed_file_path['scheme'], array( 'http', 'https', 'ftp' ), true ) ) && isset( $parsed_file_path['path'] ) ) {
			$remote_file = false;
			$file_path   = $parsed_file_path['path'];
		}

		return array(
			'remote_file' => $remote_file,
			'file_path'   => $file_path,
		);
	}
}
